"""
/***************************************************************************
Name            :  about_controller.py
Description     :  A class for about form
copyright       :  (C) 2022-2023 by FEWS
email           :  minxuansun@contractor.usgs.gov
Created         :  10/24/2022 - cholen
Modified        :  09/24/2025 - dhackman - Update logic for view_manual
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os

from PyQt5.QtWidgets import QDialog
from PyQt5 import QtGui

from fews_tools.forms.Ui_GeoCLIMAbout import Ui_GeoCLIMAbout
from fews_tools.utilities import geoclim_utilities as util


class AboutController(QDialog):
    """
    Class GeoCLIM Help About Controller
    """
    def __init__(self):
        QDialog.__init__(self)
        self.ui = Ui_GeoCLIMAbout()
        self.ui.setupUi(self)
        logos = os.path.join(
            os.path.dirname(__file__), "..", "icons", "logos.PNG")
        self.ui.lblLogos.setPixmap(QtGui.QPixmap(logos))
        self.ui.lblLogos.setScaledContents(True)

        gc_logo = os.path.join(
            os.path.dirname(__file__), "..", "icons", "GC_2.PNG")
        self.ui.lblGClogo.setPixmap(QtGui.QPixmap(gc_logo))
        self.ui.lblGClogo.setScaledContents(True)

        self.load_description()
        self.load_programmers()
        self.load_oversight()
        self.load_testers()
        self.load_version()

    def load_description(self):
        """
        Load project description
        """
        desc = ("GeoCLIM is a software tool facilitating climatological\n" +
                "analysis of historical rainfall and temperature data.  It\n " +
                "was developed by USGS FEWS NET in support of the USAID\n" +
                "PREPARED, Global Climate Change, and FEWS NET activities.")
        self.ui.lblProjectDesc.setText(desc)

    def load_oversight(self):
        """
        Load project oversight members
        """
        oversight_list = ["Chris Funk",
                          "Jim Rowland"]
        util.fill_misc_widget_no_sort(
            self.ui.projectListWidget, oversight_list)

    def load_programmers(self):
        """
        Load software programmers members
        """
        programmer_list = ["Tamuka Magadzire",
                           "Cheryl Holen",
                           "Minxuan Sun",
                           "Derek Hackman",
                           "Jacob Howton",
                           "Karthik Vanumamalai",
                           "Joshua Sickmeyer",
                           "Austin Christianson",
                           "Ashtyn Van Keulen"
                           ]
        util.fill_misc_widget_no_sort(
            self.ui.programListWidget, programmer_list)

    def load_testers(self):
        """
        Load load test members
        """
        tester_list = ["Diego Pedreros",
                       "Chris Funk",
                       "Claudia J. Young",
                       "Mario Rodriguez",
                       "Ibrahima Kadaoure"
                       ]
        util.fill_misc_widget_no_sort(
            self.ui.testListWidget, tester_list)

    def load_version(self):
        """
        Load version
        """
        version = "3.3.0"
        metadata_file = os.path.join(os.path.dirname(__file__), "metadata.txt")
        if os.path.exists(metadata_file):
            # read the version from metadata
            with open(metadata_file, 'r') as read_obj:
                for row in read_obj:
                    temp_string = row.strip()
                    if 'version=' in temp_string:
                        version = temp_string.split("=")[1]
                        break

        vers_list = [version,
                     "Copyright 2013 - 2025",
                     "All Rights Reserved"]
        util.fill_misc_widget_no_sort(self.ui.versionListWidget, vers_list)
