"""
/***************************************************************************
Name            :  about_controller.py
Description     :  A class for about form
copyright       :  (C) 2022-2023 by FEWS
email           :  minxuansun@contractor.usgs.gov
Created         :  10/24/2022 - cholen
Modified        :  09/24/2025 - dhackman - Update logic for view_manual_wrsi
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os

from PyQt5.QtWidgets import QDialog
from PyQt5 import QtGui


from fews_tools.forms.Ui_GeoWRSIAbout import Ui_GeoWRSIAbout
from fews_tools.utilities import geoclim_utilities as util


class AboutGeoWRSIController(QDialog):
    """
    Class GeoWRSI Help About Controller
    """
    def __init__(self):
        QDialog.__init__(self)
        self.ui = Ui_GeoWRSIAbout()
        self.ui.setupUi(self)
        logos = os.path.join(
            os.path.dirname(__file__), "..", "icons", "logos.PNG")
        self.ui.lblLogos.setPixmap(QtGui.QPixmap(logos))
        self.ui.lblLogos.setScaledContents(True)

        gc_logo = os.path.join(
            os.path.dirname(__file__), "..", "icons", "GeoWRSI.PNG")
        self.ui.lblGClogo.setPixmap(QtGui.QPixmap(gc_logo))
        self.ui.lblGClogo.setScaledContents(True)

        self.load_description()
        self.load_programmers()
        self.load_oversight()
        self.load_testers()
        self.load_version()

    def load_description(self):
        """
        Load project description
        """
        desc = ("Famine Early Warning Systems Network (FEWS NET) GeoWRSI " +
                "is a spatially explicit water requirements satisfaction " +
                "index(WRSI) implementation.\n" +
                "Originally developed by FAO, the WRSI has been adapted and " +
                "extended by USGS in the QGIS FEWS Tools plugin to support " +
                "FEWS NET monitoring requirments. The WRSI measures the " +
                "degree to which a crop's water requirement has been met " +
                "by available soil moisture. The USGS implementation of " +
                "WRSI is calculated using gridded precipitation, potential " +
                "evapotranspiration and soil water holding capacity.\n" +
                "The GeoWRSI was implemented by the USGS in support of the " +
                "USAID FEWS NET activity. Collaboration was provided by " +
                "the SADC Regional Remote Sensing Unit.")
        self.ui.lblProjectDesc.setText(desc)

    def load_oversight(self):
        """
        Load model development
        """
        model_development_list = ["Gabriel Senay",
                                  "James Verdin",
                                  "Bob Klaver"]
        util.fill_misc_widget_no_sort(
            self.ui.modelDevListWidget, model_development_list)

    def load_programmers(self):
        """
        Load software programmers members
        """
        programmer_list = ["Tamuka Magadzire",
                           "Chris Funk",
                           "Will Turner",
                           "Cheryl Holen",
                           "Minxuan Sun",
                           "Derek Hackman",
                           "Jacob Howton",
                           "Ashtyn Van Keulen"
                           ]
        util.fill_misc_widget_no_sort(
            self.ui.programListWidget, programmer_list)

    def load_testers(self):
        """
        Load load test members
        """
        tester_list = ["Diego Pedreros",
                       "Chris Funk",
                       "Gideon Galu",
                       "Lorena Aguilar",
                       "Alkhalil Adoum"
                       ]
        util.fill_misc_widget_no_sort(
            self.ui.testListWidget, tester_list)

    def load_version(self):
        """
        Load version
        """
        version = "3.3.0"
        metadata_file = os.path.join(os.path.dirname(__file__), "metadata.txt")
        if os.path.exists(metadata_file):
            # read the version from metadata
            with open(metadata_file, 'r') as read_obj:
                for row in read_obj:
                    temp_string = row.strip()
                    if 'version=' in temp_string:
                        version = temp_string.split("=")[1]
                        break
        vers_list = [version,
                     "Copyright 2013 - 2025",
                     "All Rights Reserved"]
        util.fill_misc_widget_no_sort(self.ui.versionListWidget, vers_list)
