'''
/***************************************************************************
Name	   :  import_crop_controller.py
Description:  Crop Import Functions for FEWSTools plugin
copyright  :  (C) 2022-2023 by FEWS
email      :  jhowton@contractor.usgs.gov
Created    :  05/16/2022 - jhowton
Modified   :  mm/dd/yyyy - <developer> - Description
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
'''

import os
from os.path import exists, splitext

from PyQt5.QtWidgets import QFileDialog, QMessageBox

from fews_tools.models.crops_model import CropsModel


def ImportCrop():
    '''
    Import a Crop File with a layout like:

    #Cassava Crop Coefficient
    cassava
    crop_F1
    0.1
    crop_F2
    0.26
    crop_F3
    0.71
    crop_K1
    0.15
    crop_K2
    0.70
    crop_K3
    0.20
    crop_r
    0.9
    crop_rini
    0.1
    crop_kp
    0.50
    crop_c1
    0.75
    crop_c2
    0.25
    '''

    crop_dic = {
        'crop_Name': '',
        'Comments': '',
        'crop_F1': '',
        'crop_F2': '',
        'crop_F3': '',
        'crop_K1': '',
        'crop_K2': '',
        'crop_K3': '',
        'crop_r': '',
        'crop_rini': '',
        'crop_kp': '',
        'crop_c1': '',
        'crop_c2': '',
        'defaultCrop': ''}

    file_name = QFileDialog.getOpenFileName(None, 'Select Crop File')

    # Check to make sure the user selected a file and it ends in ".crop"
    if exists(file_name[0]) and splitext(file_name[0])[1] == '.crop':

        # Get the crop name from the file the user selected
        _, fname = os.path.split(file_name[0])

        crop_dic['crop_Name'] = fname.split('.')[0]

        with open(file_name[0], 'r') as read_obj:
            for row in read_obj:

                if row[0] == '#':
                    crop_dic['Comments'] = str(row)

                else:
                    key = row.strip()
                    if key in crop_dic.keys():
                        value = read_obj.readline().strip()
                        crop_dic[key] = str(value)

        reply = QMessageBox.question(
            None,
            'New default crop?',
            'Do you want to set this as the default crop?',
            QMessageBox.Yes, QMessageBox.No)

        if reply == QMessageBox.Yes:
            crop_dic['defaultCrop'] = "Yes"
        else:
            crop_dic['defaultCrop'] = "No"

        crop_val_list = [
            crop_dic['crop_Name'],
            crop_dic['crop_F1'],
            crop_dic['crop_F2'],
            crop_dic['crop_F3'],
            crop_dic['crop_K1'],
            crop_dic['crop_K2'],
            crop_dic['crop_K3'],
            crop_dic['crop_r'],
            crop_dic['crop_rini'],
            crop_dic['crop_kp'],
            crop_dic['crop_c1'],
            crop_dic['crop_c2'],
            crop_dic['defaultCrop'],
            crop_dic['Comments']
        ]

        CropsModel().write_crop_record(crop_val_list)

    else:
        QMessageBox.information(None, "Invalid File",
                                "Select a crop file ending in '.crop'",
                                QMessageBox.Ok)
