'''
/***************************************************************************
Name	   :  date_utilities.py
Description:  GeoCLIM Date Utility Functions for FEWSTools plugin
copyright  :  (C) 2020-2023 by FEWS
email      :  minxuansun@contractor.usgs.gov
Created    :  12/23/2020 - cholen
Modified   :  xx/xx/xxxx - Developer - Description


 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
'''
import calendar
import datetime


def get_dekad_start_and_end_days(ds_dic, dekad_date):
    '''
    Figure out the start and end days for a pentad
    param(dic) - ds_dic - dataset_dictionary
    param(string) - pentad_date - pentad date in format YYYYPT or similar
    return(obj) - start_day datetime object
    return(obj) - end_day datetime object
    '''
    # get year
    if 'YYYY' in ds_dic['DATADATEFORMAT']:
        year_i = int(dekad_date[:4])
    else:
        year_i = int(dekad_date[:2])
        # this is a total hack but we need to deal with et data
        # nobody else still uses 2 digit years
        if year_i <= 60:
            year_i = int('20' + str(year_i))
        else:
            year_i = int('19' + str(year_i))

    # get month
    if 'MM' in ds_dic['DATADATEFORMAT']:
        month_i = int(dekad_date[-3:-1])
        dekad_of_month = int(dekad_date[-1])
    else:
        dekad_of_year = int(dekad_date[-2:])

        test = dekad_of_year % 3
        if test == 0:
            dekad_of_month = 3
        else:
            dekad_of_month = test

        month_i = int((dekad_of_year - 1) / 3) + 1

    s_day_i = int((dekad_of_month * 10) - 9)

    if dekad_of_month != 3:
        e_day_i = s_day_i + 9
    else:  # get last calendar day of the month for that year
        e_day_i = calendar.monthrange(year_i, month_i)[1]
    # year and month will be the same for both start and end
    start_day = datetime.datetime(year_i, month_i, s_day_i)
    end_day = datetime.datetime(year_i, month_i, e_day_i)

    return start_day, end_day


def get_pentad_start_and_end_days(ds_dic, pentad_date):
    '''
    Figure out the start and end days for a pentad
    param(dic) - ds_dic - dataset_dictionary
    param(string) - pentad_date - pentad date in format YYYYPT or similar
    return(obj) - start_day datetime object
    return(obj) - end_day datetime object
    '''
    # get year
    if 'YYYY' in ds_dic['DATADATEFORMAT']:
        year_i = int(pentad_date[:4])
    else:
        year_i = int(pentad_date[:2])
        # this is a total hack but we need to deal with et data
        # nobody else still uses 2 digit years
        if year_i <= 60:
            year_i = int('20' + str(year_i))
        else:
            year_i = int('19' + str(year_i))

    # get month
    if 'MM' in ds_dic['DATADATEFORMAT']:
        month_i = int(pentad_date[-3:-1])
        pentad_of_month = int(pentad_date[-1])
    else:
        pentad_of_year = int(pentad_date[-2:])

        test = pentad_of_year % 6
        if test == 0:
            pentad_of_month = 6
        else:
            pentad_of_month = test

        month_i = int((pentad_of_year - 1) / 6) + 1

    s_day_i = int((pentad_of_month * 5) - 4)
    if pentad_of_month != 6:
        e_day_i = s_day_i + 4
    else:  # get last calendar day of the month for that year
        e_day_i = calendar.monthrange(year_i, month_i)[1]
        # year and month will be the same for both start and end
    start_day = datetime.datetime(year_i, month_i, s_day_i)
    end_day = datetime.datetime(year_i, month_i, e_day_i)

    return start_day, end_day


def convert_jdate_format(jul_day_dt_obj, date_format):
    '''
    Converts from a 3 digit Julian day to a 1 to 3 digit Julian day.
    (leading zeros are discarded)
    python's datetime doesn't seem to have a format for this
    '''
    if date_format == 'special_YYYYJ':
        yr_string = jul_day_dt_obj.strftime('%Y')
    else:
        yr_string = jul_day_dt_obj.strftime('%y')

    day_3d = jul_day_dt_obj.strftime('%j')
    day_string = day_3d.lstrip('0')

    jul_day_1d_str = yr_string + day_string

    return jul_day_1d_str
