'''
/***************************************************************************
Name	   :  reclassify_grids_worker.py
Description:  Reclassify Grids tool worker
copyright  :  (C) 2019-2023 by FEWS
email      :  minxuansun@contractor.usgs.gov
Modified   :  04/10/2020 - MSun - Refine error reporting logic and
              rewrote value replace function
              01/20/2022 - cholen - New utilities, cleanup
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
'''
from qgis.core import QgsMessageLog, Qgis

from PyQt5 import QtCore

from fews_tools.utilities import geoclim_gdal_utilities as g_util


class ReclassifyGridsWorker(QtCore.QObject):
    '''
    Reclassify Grids worker class
    '''

    def __init__(self, input_file_list, output_file_list, reclassify_list):
        QtCore.QObject.__init__(self)
        self.input_file_list = input_file_list
        self.output_file_list = output_file_list
        self.reclassify_list = reclassify_list
        self.killed = False
        self.step = 0.0
        self.curr_progress = 0.0

    def run(self):
        '''
        Run function for Reclassify Grids object
        '''
        ret_tuple = None
        try:
            QgsMessageLog.logMessage(u"Begin reclassifying...",
                                     level=Qgis.Info)

            # put reclassify algorithm here
            # figure progress step for bar
            self.step = 99.0 / (len(self.input_file_list))
            self.curr_progress += self.step
            self.progress.emit(int(self.curr_progress))

            for entry in self.input_file_list:
                temp_index = self.input_file_list.index(entry)
                output_filename = self.output_file_list[temp_index]
                _, _, _, geoxfrm, data_type_name = g_util.get_geotiff_info(entry)
                # Process calculation with input extent and resolution
                err = g_util.replace_value(
                    geoxfrm,
                    self.reclassify_list,
                    entry,
                    output_filename,
                    data_type_name)
                if self.killed is True:
                    raise KeyboardInterrupt
                if err is True:
                    raise RuntimeError
                self.curr_progress += self.step
                self.progress.emit(int(self.curr_progress))

            if self.killed is False:
                self.progress.emit(100)
                ret_tuple = (0, "Reclassify Grids complete")
        # exit with appropriate message on killed (KeyboardInterrupt)
        except KeyboardInterrupt:
            self.progress.emit(0)
            ret_tuple = (0, u"Reclassify Grids aborted by user")
        # forward any execeptions upstream
        except BaseException as exc:
            self.error.emit(exc, u"Unspecified error in Reclassify Grids")
        self.finished.emit(ret_tuple)

    def kill(self):
        '''
        Set the kill flag
        '''
        self.killed = True

    finished = QtCore.pyqtSignal(object)

    error = QtCore.pyqtSignal(Exception, str)

    progress = QtCore.pyqtSignal(int)
